<?php defined('BASEPATH') OR exit('No direct script access allowed');
/**
* @package direct-as-a-service
* @subpackage controllers
* @filesource
*//** */
 
require_once 'mailbox_controller.php';

/**
* API methods to manage a mailbox's messages.
*
* Inherits mailbox validation from the mailbox controller.  Mailbox name is a required field for all actions on this controller, with the exception of messages to send or save a messagage; these actions require the sender's email address to be provided.
*
* @author           <bell_adam@bah.com>
* @author M. Gibbs <gibbs_margaret@bah.com>
* @author Elan Jaffee <jaffee_elan@bah.com>
*
* @package direct-as-a-service
* @subpackage controllers
*/
class Message_retrieve_controller extends Mailbox_controller{	
	
	/**
	* Everything that applies to all actions should go here.
	*/
	function __construct(){
		parent::__construct();
		$this->load->helper(array('mail', 'validate'));
		$this->load->model('messagestatusmodel');
	}	
	
	public function count_get(){
		$this->respond_with_error_if_fields_are_missing();
		$this->respond_with_error_if_user_is_unauthorized('retrieve');
		
		//SET OPTIONAL FIELDS && CHECK FOR INVALID FIELD VALUES
		$folder = element('folder', $this->get(), 'all');
		
		//folder should either be a Folder entity or the name of a known location		
		if(Folder::formatted_like_an_id($folder)) 
			$folder = first_element($this->mailbox->folders( array('id' => $folder))) ;
		if(!Folder::is_an_entity($folder) && !in_array($folder, $this->valid_locations())){
			$this->invalid_fields[] = 'folder';
		}
		
		$filter = element('filter', $this->get(), null);
		if(!empty($filter))
			$filter =  base64_decode(rawurldecode($filter));

		//if there is an error with the passed in fields create an error message
		$this->respond_with_error_if_fields_are_invalid();
		$this->apply_filter($filter);
		
		if(Folder::is_an_entity($folder)){
			$total = $folder->message_count();
		}else{
			$count_method = $folder.'_message_count';
			$count_method = strip_from_beginning('all_', $count_method); //all is a weird case, adjust accordingly
			$count_method = str_replace('draft_message', 'drafts', $count_method); //drafts -> also a weird case
			$total = $this->mailbox->$count_method;
		}
		
		$this->response_message['count'] = $total;
		$this->response($this->response_message, 200);
	}
	
	/**
	* Retrieve a single message.
	*
	* This action is available to both active and inactive mailboxes.
	*
	* Required fields:
	* 	- id (message id)
	* 	- mailbox (mailbox name)
	* 
	* Optional fields:
	*	- mark ('read', 'unread')
	* 	- part (see the array keys of Message::$parts for options)
	*
	*/
	public function message_get(){
		if(!array_key_exists('id', $this->get()))
			$this->missing_required_fields[] = 'id';
		
		$this->respond_with_error_if_fields_are_missing(); //mailbox is only the required parameter - it will be set in the constructor
		$this->respond_with_error_if_user_is_unauthorized('retrieve');		
	
		$id = $this->get('id');
		if(!Message::formatted_like_an_id($id)) $this->invalid_fields[] = 'id';	
	
		//validate input		
		$mark = $this->get('mark');
		if($mark && $mark != 'read' && $mark != 'unread') $this->invalid_fields[] = 'mark';
		
		$part = $this->get('part');
		if(!empty($part) && !Message::is_a_part($part)) $this->invalid_fields[] = 'part';	
		
		$this->respond_with_error_if_fields_are_invalid();

		//find the message
		$message = Message::find_one($id);
		if(!Message::is_an_entity($message) || !$message->belongs_to_mailbox($this->mailbox)){
			$this->response('Message not found.', 422);
		}			
	
		//if needed, mark the value read/unread	
		if($message->is_incoming()){
			if($mark == 'read') $message->mark_as_read();
			if($mark == 'unread') $message->mark_as_unread();
		}

#TODO - CHECK TO SEE IF IT WILL CAUSE WEBMAIL PROBLEMS IF WE SWITCH THIS TO A SINGULAR ID			
		$this->response_message['ids'] = array($id);
		
#TODO - IF WE KEEP PLURAL IDS, THIS SHOULD BE ARRAY($VALUES) FOR CONSISTENCY.  BUT WE PROBABLY SHOULDN'T KEEP THIS TO PLURAL IDS	
		$this->response_message['mail'] = $message->values_for_api($part);
		$this->response($this->response_message, 200);	
	}
	
	/**
	* Retrieve messages from a mailbox.	
	*
	* This action is available to both active and inactive mailboxes.
	*
	* Required parameters:
	* 	- mailbox (the mailbox name: this will be verified in the constructor & the mailbox object will be saved to {@link mailbox})
	*
	*/
	public function messages_get(){
		$this->respond_with_error_if_fields_are_missing(); //mailbox is only the required parameter - it will be set in the constructor
		$this->respond_with_error_if_user_is_unauthorized('retrieve');		
		
		if(array_key_exists('id', $this->get()))
			return $this->message_get(); //essentially, redirect to message_get
		

		//FIND THE OPTIONAL FIELD VALUES
		//listing & showing the optional fields with their default values makes it easier to see what options are available
		$optional_fields = array( 'folder' => 'inbox', 
								  'mark' => false,
								  'part' => '',
								  'filter' => null,
								  'limit' => MAX_MESSAGES,
								  'order_by' => 'timestamp',
								  'start' => 0,
								  'order' => 'desc');			  
		
		//set variables for each optional field						  
		foreach($optional_fields as $optional_field => $default_value){
			$$optional_field = element($optional_field, $this->get(), $default_value); //checks to see if $field is in $get - if not, sets to $default_value
		}
		
		//VALIDATE THE OPTIONAL FIELDS AS NEEDED
		
		//folder should either be a Folder entity or the name of a known location		
		if(Folder::formatted_like_an_id($folder)) 
			$folder = Folder::find_one($folder);
		if(!Folder::is_an_entity($folder) && !in_array($folder, $this->valid_locations())){
			$this->invalid_fields[] = 'folder';
		}
		if($folder == 'all') $folder = '';
				
		//get part of message.  If left not set returns all
		if(!empty($part) && !Message::is_a_part($part)){
			$this->invalid_fields[] = 'part';
		}
		if(!($filter)){
			$filter = base64_decode(rawurldecode($filter));
		}
		if($mark && $mark !== 'read' && $mark !== 'unread'){
			$this->invalid_fields[] = 'mark';
		}
				
		//get limit for amount of messages returned.  can't be over constant MAX_MESSAGES
		if(!$this->is->unsigned_integer($limit) || $limit > MAX_MESSAGES)
			$this->invalid_fields[] = 'limit';
			
		//get the offset.  0 = no offset
		if(!$this->is->unsigned_integer($start)){
			$this->invalid_fields[] = 'start';
		}

		//order by.  default is timestamp.  can only order by set parameters
		if(!in_array($order_by, array('timestamp', 'id', 'sender', 'to','cc','bcc','attachments','size','subject','plain'))){
			$this->invalid_field[] =  'order_by';
		}
		//order.  
		if($order !== 'desc' && $order !== 'asc'){
			$this->invalid_fields[] = 'order';
		}
		
		$this->respond_with_error_if_fields_are_invalid();
		
		$filter = element('filter', $this->get(), null);
		if(!empty($filter))
			$filter =  base64_decode(rawurldecode($filter));
		
		// ASSEMBLE THE DATABASE QUERY
		$this->apply_filter($filter);
		if ($order_by === 'subject') {
			$subject_without_prefix = 'subject';
			$reply_forward_prefixes = array('RE: ', 'RE:', 'Re: ', 're: ', 'Re:', 're:', 'FWD: ', 'FWD:' , 'Fwd: ', 'fwd: ', 'Fwd:', 'fwd:', 'Undeliverable: ');
			foreach($reply_forward_prefixes as $prefix) {
				$subject_without_prefix = "replace(" . $subject_without_prefix . ",'" . $prefix . "','')";
			}
			Message::db()->select($subject_without_prefix . ' AS _subject_without_prefix', $escape = FALSE);
			//turn protect identifiers off so order_by doesn't try to escape the calculated _subject_without_prefix value
			//note that most active record methods accept a parameter that lets you indicate that the value shouldn't be escaped. But for some reason, order_by doesn't.  Make sure to use the parameter when it's available, like for where.() -- MG
			$original_protect_identifiers_value = Message::db()->_protect_identifiers;
			Message::db()->_protect_identifiers = false;
			Message::db()->order_by("replace(replace(replace(replace(replace(replace(replace(replace(replace(replace(replace(replace(replace(subject,'RE: ', ''), 'RE:', ''), 'Re: ', ''), 're: ', ''), 'Re:', ''), 're:', ''), 'FWD: ', ''), 'FWD:', ''), 'Fwd: ', ''), 'fwd: ', ''), 'Fwd:', ''), 'fwd:', ''), 'Undeliverable: ', '')",$order);
			Message::db()->_protect_identifiers = $original_protect_identifiers_value;
		}
		
		Message::db()->order_by($order_by,$order);
		Message::db()->limit($limit,$start);
				
		if(is_string($folder)){
			//message methods follow a common pattern, so we can generate them instead of needing to hard code separate cases
			//following a DRY approach will make the code easier to maintain if we need to make changes later on
			$relationship_name = $folder;
			if($folder != 'draft') $relationship_name = implode_nonempty( '_', array($relationship_name, 'message') );
			$message_method = plural( $relationship_name );
			$message_count_method = $relationship_name.'_count';
			$messages = $this->mailbox->$message_method();
			$this->apply_filter($filter);
			$total = $this->mailbox->$message_count_method();
			$this->apply_filter($filter);
			$unseen = $this->mailbox->$message_count_method( array('seen' => false));
		}
		else{
			$messages = $folder->messages();
			$this->apply_filter($filter);
			$total = $folder->message_count();
			$this->apply_filter($filter);
			$unseen = $folder->message_count(array('seen' => false));
		}
		
		$this->response_message['total'] = $total;
		$this->response_message['total_unseen'] = $unseen;
		$this->response_message['count'] = count($messages);
		$this->response_message['ids'] = array_keys($messages);
		
		$this->response_message['mail'] = array();
		foreach($messages as $message){
			if($message->is_incoming()){
				if($mark == 'read') $message->mark_as_read();
				if($mark == 'unread') $message->mark_as_unread();
			}
				
			$this->response_message['mail'][$message->id] = $message->values_for_api($part);
		}
		$this->response($this->response_message, 200);
	}
	
///////////////////////////////////////////////
// HELPER METHODS
///////////////////////////////////////////////	
	
	protected function apply_filter($filter){
		if(empty($filter)) return;
		if(!$this->json->is_valid($filter)) return $this->response('Filter is not a proper JSON', 422);
		$filters = $this->json->decode($filter); //don't try to suppress errors - if we get an error while decoding valid JSON, we want to know about it.
		if(empty($filters)) return;
				

		$where = array('seen','flags','size','sender','to','cc','bcc','subject','plain','html','priority');
        foreach($where as $field){
			if(isset($filters->$field)){
				Message::db()->where($field , $filters->$field);
			}
        }
   
        if(isset($filters->original_folder) && $filters->original_folder != ''){
        	$possible_folders = array('inbox','sent','draft');
        	$original_folder_array = explode(',',  $filters->original_folder);
        	$allowed_folders = array_intersect($possible_folders, $original_folder_array);
        	$num_folders = count($allowed_folders);
        	if($num_folders !== 3){ // not have all
        		if($num_folders === 2){// two folders
        			if(in_array('inbox', $allowed_folders)){
        				if(in_array('sent', $allowed_folders))
        					Message::db()->where('draft',0);
        				if(in_array('draft', $allowed_folders))
        					Message::db()->where('sent',0);
        			}
        			else{ //sent and draft
        				Message::db()->where("(([draft] = '0' AND [sent] = '1') OR ([draft] = '1' AND [sent] = '0'))");
        			}
        		}
        		else if ($num_folders === 1){ // only one
        			switch (reset($allowed_folders)){
        				case 'inbox':
        					Message::db()->where("sent","0");
        					Message::db()->where("draft","0");
        					break;
        				case 'draft':
        					Message::db()->where("sent","0");
        					Message::db()->where("draft","1");
        					break;
        				case 'sent':
        					Message::db()->where("sent","1");
        					Message::db()->where("draft","0");
        					break;
        			}
        		}
        	}
        }
        
		if(isset($filters->first_date)){
			Message::db()->where('timestamp >=', $filters->first_date);
		}
		if(isset($filters->end_date)){
			Message::db()->where('timestamp <=', $filters->end_date);
		}
		if(isset($filters->date)){
			Message::db()->where('timestamp' , $filters->date);
		}
		if(isset($filters->has_attachment)){
			if($filters->has_attachment){
				Message::db()->where('attachments !=',  "{}");
			}
			else{
				Message::db()->where('attachments',  "{}");
			}
		}
		if(isset($filters->smallest)){
			Message::db()->where('size >=', $filters->smallest);
		}
		if(isset($filters->largest)){
			Message::db()->where('size <=', $filters->largest);
		}
		if(isset($filters->body)){
			$v = Message::db()->escape_like_str($filters->body);
			Message::db()->where("([plain]='{$v}' OR [html]='{$v}')");
		}
		$like = array('flags','sender','to','cc','bcc','subject','plain','html','attachments');
        foreach($like as $field){
			$contains = "contains_". $field;
			if(isset($filters->$contains)){
				Message::db()->like($field , $filters->$contains);
			}
        }
		if(isset($filters->contains_body)){
			$v = Message::db()->escape_like_str($filters->contains_body);
			Message::db()->where("([plain] like'%{$v}%' OR [html] like'%{$v}%')");
		}
		if(isset($filters->contains_recipients)){
			$v = Message::db()->escape_like_str($filters->contains_recipients);
			Message::db()->where("([to] like'%{$v}%' OR [cc] like'%{$v}%' OR [bcc] like'%{$v}%')");
		}
		if(isset($filters->fuzzy)){
			$v = Message::db()->escape_like_str($filters->fuzzy);
			Message::db()->where("([to] like'%{$v}%' OR [cc] like'%{$v}%' OR [bcc] like'%{$v}%' OR [subject] like'%{$v}%' OR [html] like'%{$v}%' OR [plain] like'%{$v}%' OR [sender] like'%{$v}%' OR [attachments] like '%{$v}%')");
		}
	}
	
}
?>